<?php
/**
 * @package     e4j.VikUpdater
 * @subpackage  Library.Network
 * @copyright   Copyright (C) 2023 e4j - Extensionsforjoomla.com. All Rights Reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace E4J\VikUpdater\Network;

// No direct access to this file
defined('_JEXEC') or die('Restricted access');

/**
 * This class extends the functions of the JHttp native object.
 *
 * @since 1.0
 * @since 1.2.2  This class doesn't extend JHttp anymore because of a discrepancy
 *               between the methods declaration in J3 and J4.
 * @since 1.3    Renamed from VikHttp.
 * @since 1.4    Renamed from VUPNetworkHttp
 */
class Http
{
	/**
	 * Http transporter instance.
	 * 
	 * @var JHttp
	 * @since 1.2.2
	 */
	protected $http;

	/**
	 * Class constructor.
	 *
	 * @uses 	isSupported() 	Check if the transport is supported.
	 *
	 * @throws 	Exception 	In case there is no connection driver available.
	 */
	public function __construct()
	{
		if (!$this->isSupported())
		{
			throw new \Exception('No connection driver is supported from the server.');
		}

		// create new HTTP transport
		$this->http = new \JHttp();

		/**
		 * Disable SSL validation since we are always contacting a certified end-point.
		 * 
		 * @since 1.2.2
		 */
		$this->http->setOption('transport.curl', [
			CURLOPT_SSL_VERIFYHOST => 0,
			CURLOPT_SSL_VERIFYPEER => 0,
		]);
	}

	/**
	 * Method to send the GET command to the server.
	 *
	 * @param   string  $url      Path to the resource.
	 * @param   array   $headers  An array of name-value pairs to include in the header of the request.
	 * @param   int     $timeout  Read timeout in seconds.
	 *
	 * @return  JHttpResponse
	 */
	public function get($url, array $headers = [], $timeout = null)
	{
		return $this->http->get(self::END_POINT . $url, $headers, $timeout);
	}

	/**
	 * Method to send the POST command to the server.
	 *
	 * @param   string  $url      Path to the resource.
	 * @param   mixed   $data     Either an associative array or a string to be sent with the request.
	 * @param   array   $headers  An array of name-value pairs to include in the header of the request
	 * @param   int     $timeout  Read timeout in seconds.
	 *
	 * @return  JHttpResponse
	 */
	public function post($url, $data, array $headers = [], $timeout = null)
	{
		return $this->http->post(self::END_POINT . $url, $data, $headers, $timeout);		
	}

	/**
	 * Check if the connection driver is supported.
	 *
	 * @return 	bool  True if it is possible to perform connections, otherwise false.
	 */
	public function isSupported()
	{
		return true;
	}

	/**
	 * End-Point identifier.
	 *
	 * @var string
	 * 
	 * @since 1.3 changed from "https://extensionsforjoomla.com/api/vqs/"
	 * @since 1.4 changed from "https://extensionsforjoomla.com/api/updates/"
	 */
	const END_POINT = 'https://extensionsforjoomla.com/vikapi?task=joomla.updateserver';
}
