<?php
/**
 * @package     e4j.VikUpdater
 * @subpackage  Library.Software
 * @copyright   Copyright (C) 2023 e4j - Extensionsforjoomla.com. All Rights Reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace E4J\VikUpdater\Software;

// No direct access to this file
defined('_JEXEC') or die('Restricted access');

/**
 * This class wraps the major info of a software version.
 *
 * @since 1.0
 * @since 1.3  Renamed from SoftwareVersion.
 * @since 1.4  Renamed from VUPSoftwareVersion.
 */
abstract class Version implements \JsonSerializable
{
	/**
	 * A list of properties related to the software version.
	 *
	 * @var array
	 */
	private $args = null;

	/**
	 * Class constructor.
	 *
	 * @param 	array|object  $args  The list of the object that contains the version info.
	 */
	public function __construct($args)
	{
		$this->args = array();

		if (is_array($args) || is_object($args))
		{
			foreach ($args as $k => $v)
			{
				$this->args[$this->cleanProperty($k)] = $v;
			}
		}
	}

	/**
	 * Magic method to get properties of the object.
	 *
	 * @param   string 	$name       The name of the function. The string next to "get" word will be the name of the property.
	 *								For example, getVersion will retrieve the "version" property.
	 * 								Otherwise it is possible to add properties by using "set" as first 3 chars.
	 * 								For example, setVersion will add the "version" property.
	 *
	 * @param   array  	$arguments 	Array containing arguments to retrieve the setting.
	 *
	 * @return  mixed 	The value of the property or this class to support chaining.
	 */
	public function __call($name, $arguments)
	{
		if (substr($name, 0, 3) == 'get')
		{
			if (strlen($name) > 3)
			{
				$prop = $this->cleanProperty(substr($name, 3));
			}
			else
			{
				$prop = isset($arguments[0]) ? $arguments[0] : '';
			}

			if (isset($this->args[$prop]))
			{
				return $this->args[$prop];
			}

		} 
		else if (substr($name, 0, 3) == 'set')
		{
			if (strlen($name) > 3)
			{
				$prop = $this->cleanProperty(substr($name, 3));
			}
			else
			{
				$prop = isset($arguments[0]) ? $arguments[0] : '';
			}

			$this->args[$prop] = isset($arguments[1]) ? $arguments[1] : (isset($arguments[0]) ? $arguments[0] : '');

			return $this;
		}

		return null;
	}

	/**
	 * Make safe the specified property name.
	 *
	 * @param 	string  $prop  The name of the property to make safe.
	 *
	 * @return 	string 	The safe property name.
	 */
	protected function cleanProperty($prop)
	{
		return strtolower(str_replace('_', '', $prop));
	}

	/**
	 * Return the properties of the version as an array.
	 *
	 * @return 	array  The properties list.
	 */
	public function toArray()
	{
		return $this->args;
	}

	/**
	 * Return the properties of the version as a JSON string.
	 *
	 * @return 	string 	The properties list in JSON.
	 * 
	 * @deprecated 1.6  Directly use the `json_encode` function.
	 */
	public function toJson()
	{
		return json_encode($this->toArray());
	}

	/**
	 * Creates a standard object, containing all the supported properties,
	 * to be used when this class is passed to "json_encode()".
	 *
	 * @return  object
	 *
	 * @since 	1.4
	 *
	 * @see     JsonSerializable
	 */
	#[\ReturnTypeWillChange]
	public function jsonSerialize()
	{
		return $this->toArray();
	}

	/**
	 * Get a unique digest hash key of the properties.
	 *
	 * @return 	string 	The digest string.
	 */
	public function digest()
	{
		return md5(serialize($this->toArray()));
	}

	/**
	 * Check if the specified version is valid or not.
	 *
	 * @return 	bool  True if the instance is valid, otherwise false.
	 * 
	 * @since   1.4  The method has been integrated from VUPSoftwareIntegrity.
	 */
	abstract public function check();
}
